"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAgendamento = void 0;
const express_validator_1 = require("express-validator");
const agendamento_1 = __importDefault(require("../models/agendamento"));
const agenda_1 = __importDefault(require("../models/agenda"));
const main_1 = require("../util/main");
const postmark_1 = __importDefault(require("../services/postmark"));
// Metodo Para criar agendamento
const createAgendamento = async (req, res, next) => {
    try {
        const { agendaId, email, nome, tel, assunto, resumo } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const agenda = await agenda_1.default.findById(agendaId).populate({
            path: 'membro',
            populate: { path: 'dadosPessoais' },
        });
        if (!agenda) {
            const error = new Error('Agenda não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        if (agenda.status !== 'Disponivel') {
            const error = new Error(`Agenda ${agenda.status}`);
            error.statusCode = 422;
            throw error;
        }
        const agendamentoObj = {
            nome: nome,
            tel,
            assunto,
            resumo,
        };
        if (Boolean(email)) {
            agendamentoObj.email = email;
        }
        const agendamento = new agendamento_1.default(agendamentoObj);
        await agendamento.save();
        agenda.agendamento = agendamento.id;
        agenda.status = 'Indisponivel';
        await agenda.save();
        const membro = agenda === null || agenda === void 0 ? void 0 : agenda.membro;
        const dadosPessoais = membro === null || membro === void 0 ? void 0 : membro.dadosPessoais;
        if (email) {
            await postmark_1.default.sendEmail({
                To: email,
                From: process.env.SENDER_EMAIL,
                Subject: `Agendamento - ${agenda.dia}/${agenda.mes}/${agenda.ano} ${agenda.horaInicio}:${agenda.minutoInicio}-${agenda.horaTermino}:${agenda.minutoTermino}`,
                HtmlBody: `<h1>Confirmação de Agendamento</h1>
          <p>
            Parabéns o seu agendamento para o dia ${agenda.dia} de ${(0, main_1.getMes)(agenda.mes)} 
            as ${agenda.horaInicio}:${agenda.minutoInicio} foi criado com sucesso. 
            Terá a sua adiência com o ${dadosPessoais.genero === 'M' ? 'Conselheiro' : 'conselheira'} ${dadosPessoais.nome}. 
            Qualquer alteração será previamento avisado!
          </p>
        `,
                MessageStream: 'outbound'
            });
        }
        await postmark_1.default.sendEmail({
            To: dadosPessoais.email,
            From: process.env.SENDER_EMAIL,
            Subject: `Agendamento - ${agenda.dia}/${agenda.mes}/${agenda.ano} ${agenda.horaInicio}:${agenda.minutoInicio}-${agenda.horaTermino}:${agenda.minutoTermino}`,
            HtmlBody: `<h1>Confirmação de Agendamento</h1>
        <p>
          O agendamento para o dia ${agenda.dia} de ${(0, main_1.getMes)(agenda.mes)} 
          as ${agenda.horaInicio}:${agenda.minutoInicio} foi preenchido por <b>${agendamento.nome}</b>,
          para abordar sobre <b>${agendamento.assunto}</b>.
        </p>

        <h2>Contactos do solicitante</h2>
        <p>Tel: ${agendamento.tel}</p>
        ${Boolean(email) && `<p>Email: ${agendamento.email}</p>`} 
      `,
            MessageStream: 'outbound'
        });
        res.status(201).json({
            msg: 'Sucesso!',
            agendamento,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createAgendamento = createAgendamento;
