"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.addRestricaoMedicamento = exports.deleteMedicamento = exports.getOneMedicamento = exports.getAllMedicamentos = exports.createMedicamento = void 0;
const express_validator_1 = require("express-validator");
const mongoose_1 = __importDefault(require("mongoose"));
const medicamento_1 = __importDefault(require("../models/medicamento"));
const restricao_1 = __importDefault(require("../models/restricao"));
const createMedicamento = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const { nomeGenerico, codNacional, codUNSPSC, apresentacao, categoria } = req.body;
        const verifyMedicamento = await medicamento_1.default.findOne({ nomeGenerico });
        if (verifyMedicamento) {
            const error = new Error(`O medicamento com o nome ${verifyMedicamento.nomeGenerico} já foi criado!`);
            error.statusCode = 422;
            throw error;
        }
        const verifyCodUNSPSC = await medicamento_1.default.findOne({ codUNSPSC });
        if (verifyCodUNSPSC) {
            const error = new Error(`O medicamento com o código ${verifyCodUNSPSC.codUNSPSC} já foi criado!`);
            error.statusCode = 422;
            throw error;
        }
        const verifyCodNacional = await medicamento_1.default.findOne({ codNacional });
        if (verifyCodNacional) {
            const error = new Error(`O medicamento com o código ${verifyCodNacional.codNacional} já foi criado!`);
            error.statusCode = 422;
            throw error;
        }
        // Criar novO Medicamento
        const medicamento = new medicamento_1.default({
            nomeGenerico,
            codUNSPSC,
            codNacional,
            apresentacao,
            categoria
        });
        await medicamento.save();
        res.status(201).json({
            msg: `Medicamento criado com sucesso`,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createMedicamento = createMedicamento;
const getAllMedicamentos = async (req, res, next) => {
    try {
        const medicamentos = await medicamento_1.default.find({})
            .populate('categoria')
            .populate('restricoes');
        if (medicamentos.length === 0) {
            return res.status(404).json({
                success: false,
                medicamentos
            });
        }
        res.status(200).json({
            success: true,
            medicamentos
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getAllMedicamentos = getAllMedicamentos;
const getOneMedicamento = async (req, res, next) => {
    try {
        const id = req.params.id;
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (validId) {
            const medicamento = await medicamento_1.default.findOne({ _id: id });
            if (!medicamento) {
                return res.status(404).json({
                    success: false,
                    medicamento
                });
            }
            res.status(200).json({
                success: true,
                medicamento
            });
        }
        else {
            return res.status(400).json({
                success: false,
                message: 'Id inválido.'
            });
        }
    }
    catch (err) {
        next(err);
    }
};
exports.getOneMedicamento = getOneMedicamento;
const deleteMedicamento = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const id = req.params.id;
        // Verifica se o id é válido
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (!validId) {
            return res.status(400).json({
                success: false,
                message: 'ID inválido.',
            });
        }
        // Busca o Medicamento para garantir que ele existe
        const medicamento = await medicamento_1.default.findById(id);
        if (!medicamento) {
            return res.status(404).json({
                success: false,
                message: 'Medicamento não encontrado.',
            });
        }
        await medicamento_1.default.findByIdAndDelete(id);
        res.status(200).json({
            success: true,
            message: `Medicamento ${medicamento.nomeGenerico} deletada com sucesso.`,
        });
    }
    catch (err) {
        next(err); // Passa o erro para o middleware de erro
    }
};
exports.deleteMedicamento = deleteMedicamento;
const addRestricaoMedicamento = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const { id, restricao } = req.params;
        // Verifica se o id é válido
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (!validId) {
            return res.status(400).json({
                success: false,
                message: 'ID inválido.',
            });
        }
        const validRestricaoId = mongoose_1.default.Types.ObjectId.isValid(restricao) ? restricao : null;
        if (!validRestricaoId) {
            return res.status(400).json({
                success: false,
                message: 'ID da restrição inválido.',
            });
        }
        // Busca o Medicamento para garantir que ele existe
        const medicamento = await medicamento_1.default.findById(id);
        if (!medicamento) {
            return res.status(404).json({
                success: false,
                message: 'Medicamento não encontrado.',
            });
        }
        const verifyRestricao = await restricao_1.default.findOne({ _id: restricao });
        if (!verifyRestricao) {
            return res.status(404).json({
                success: false,
                message: 'Restrição não encontrada.',
            });
        }
        const restricaoMedicamento = await medicamento_1.default.findOne({ _id: id, restricoes: validRestricaoId });
        if (restricaoMedicamento) {
            return res.status(409).json({
                success: false,
                message: 'Essa restrição já se encontra neste medicamento.',
            });
        }
        // Atualiza o medicamento, adicionando a restrição ao array 'restricoes'
        await medicamento_1.default.updateOne({ _id: id }, { $push: { restricoes: verifyRestricao._id } });
        res.status(200).json({
            success: true,
            message: `Restrição adicionada ao medicamento com sucesso.`,
        });
    }
    catch (err) {
        next(err); // Passa o erro para o middleware de erro
    }
};
exports.addRestricaoMedicamento = addRestricaoMedicamento;
