"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteNoticia = exports.changeImage = exports.editNoticia = exports.getNoticia = exports.getNoticias = exports.createNoticia = void 0;
const noticia_1 = __importDefault(require("../models/noticia"));
const express_validator_1 = require("express-validator");
const file_1 = require("../util/file");
const aws_1 = require("../util/aws");
const user_1 = __importDefault(require("../models/user"));
const pagination_1 = require("../util/pagination");
const main_1 = require("../util/main");
// Controller para criar noticia
const createNoticia = async (req, res, next) => {
    try {
        let { titulo, subTitulo, fonte, link, conteudo, destaque, categoria, userId, status } = req.body;
        const imagem = req.file;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        if (!imagem) {
            const error = new Error('Faça o upload do imagem!');
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        imagem.filename = (0, file_1.generateFileName)(imagem);
        const noticia = new noticia_1.default({
            titulo,
            fonte,
            link,
            conteudo,
            imagem: imagem.filename,
            admin: userId,
            ordem: '63ab278e65d41e8e1cfa9e7a',
        });
        if (Boolean(subTitulo)) {
            noticia.subTitulo = subTitulo;
        }
        if (Boolean(categoria)) {
            noticia.categoria = categoria;
        }
        if (Boolean(status)) {
            noticia.status = status;
        }
        if (Boolean(destaque)) {
            destaque = JSON.parse(destaque);
            noticia.destaque = destaque;
            const noticiasEmDestaque = await noticia_1.default.find({ destaque: destaque });
            for (let noticia of noticiasEmDestaque) {
                noticia.destaque = false;
                await noticia.save();
            }
        }
        await noticia.save();
        await (0, aws_1.uploadFile)(imagem);
        if (admin) {
            let action = 'Criou notícia';
            await (0, main_1.createHistoricoActividade)(action, 'Noticia', noticia.id, admin);
        }
        res.status(201).json({ msg: 'Noticia registrado com sucesso', noticia });
    }
    catch (err) {
        next(err);
    }
};
exports.createNoticia = createNoticia;
// Metodo que busca todas as noticias
const getNoticias = async (req, res, next) => {
    try {
        const status = req.query.status;
        const filterObj = {};
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        if (Boolean(status)) {
            filterObj.status = status;
        }
        const total = await noticia_1.default.find(filterObj).count();
        const noticias = await noticia_1.default.find(filterObj)
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage)
            .sort({ createdAt: -1 });
        for (const noticia of noticias) {
            if (noticia.imagem) {
                noticia.imagem = await (0, aws_1.getURL)(noticia.imagem);
            }
        }
        res.status(200).json({
            msg: 'Sucesso',
            noticias,
            ...(0, pagination_1.paginationResult)(total, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getNoticias = getNoticias;
// Metodo que busca pelo id noticia
const getNoticia = async (req, res, next) => {
    try {
        const { id } = req.params;
        const noticia = await noticia_1.default.findById(id);
        if (!noticia) {
            const error = new Error('Notícia não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (noticia.imagem) {
            noticia.imagem = await (0, aws_1.getURL)(noticia.imagem);
        }
        res.status(200).json({
            msg: 'Sucesso',
            noticia,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getNoticia = getNoticia;
// Controller para editar noticia
const editNoticia = async (req, res, next) => {
    try {
        let { titulo, subTitulo, fonte, link, conteudo, categoria, destaque, userId, noticiaId, status } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const noticia = await noticia_1.default.findById(noticiaId);
        if (!noticia) {
            const error = new Error('Notícia não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (Boolean(titulo)) {
            noticia.titulo = titulo;
        }
        if (Boolean(fonte)) {
            noticia.fonte = fonte;
        }
        if (Boolean(link)) {
            noticia.link = link;
        }
        if (Boolean(conteudo)) {
            noticia.conteudo = conteudo;
        }
        if (Boolean(subTitulo)) {
            noticia.subTitulo = subTitulo;
        }
        if (Boolean(status)) {
            noticia.status = status;
        }
        if (Boolean(categoria)) {
            noticia.categoria = categoria;
        }
        if (Boolean(destaque)) {
            const noticiasEmDestaque = await noticia_1.default.find({ destaque: destaque });
            for (let noticia of noticiasEmDestaque) {
                noticia.destaque = false;
                await noticia.save();
            }
            destaque = JSON.parse(destaque);
            noticia.destaque = destaque;
        }
        await noticia.save();
        if (admin) {
            let action = 'Editou notícia';
            await (0, main_1.createHistoricoActividade)(action, 'Noticia', noticia.id, admin);
        }
        res.status(201).json({ msg: 'Sucesso', noticia });
    }
    catch (err) {
        next(err);
    }
};
exports.editNoticia = editNoticia;
const changeImage = async (req, res, next) => {
    try {
        const { noticiaId } = req.body;
        const imagem = req.file;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        if (!imagem) {
            const error = new Error('Por favor fornceça uma imagem');
            error.statusCode = 422;
            throw error;
        }
        const noticia = await noticia_1.default.findById(noticiaId);
        if (!noticia) {
            const error = new Error('Notícia não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        // Primeiro deleta o arquivo
        await (0, aws_1.deleteFile)(noticia.imagem);
        // Depois renomeia e faz o upload
        imagem.filename = (0, file_1.generateFileName)(imagem);
        await (0, aws_1.uploadFile)(imagem);
        noticia.imagem = imagem.filename;
        await noticia.save();
        noticia.imagem = await (0, aws_1.getURL)(noticia.imagem);
        res.status(201).json({
            msg: 'Sucesso!',
            noticia,
        });
    }
    catch (error) {
        next(error);
    }
};
exports.changeImage = changeImage;
const deleteNoticia = async (req, res, next) => {
    try {
        const { id, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const user = await user_1.default.findOne({ _id: userId, isAdmin: true });
        if (!user) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const noticia = await noticia_1.default.findById(id);
        if (!noticia) {
            const error = new Error('Notícia não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        await noticia.delete();
        await (0, main_1.createHistoricoActividade)(`Deletou noticia: ${noticia.titulo}`, 'Noticia', noticia.id, user);
        res.status(204).json({ msg: 'Sucesso!' });
    }
    catch (err) {
        next(err);
    }
};
exports.deleteNoticia = deleteNoticia;
