"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteCarousel = exports.changeImage = exports.editCarousel = exports.getCarousel = exports.getCarousels = exports.createCarousel = void 0;
const carousel_1 = __importDefault(require("../models/carousel"));
const express_validator_1 = require("express-validator");
const file_1 = require("../util/file");
const aws_1 = require("../util/aws");
const user_1 = __importDefault(require("../models/user"));
const pagination_1 = require("../util/pagination");
const main_1 = require("../util/main");
// Controller para criar carousel
const createCarousel = async (req, res, next) => {
    try {
        let { titulo, descricao, userId, } = req.body;
        const imagem = req.file;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        if (!imagem) {
            const error = new Error('Por favo, carregue uma imagem!');
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        imagem.filename = (0, file_1.generateFileName)(imagem);
        const carousel = new carousel_1.default({
            titulo,
            descricao,
            imagem: imagem.filename,
            admin: userId,
        });
        await carousel.save();
        await (0, aws_1.uploadFile)(imagem);
        // HistoricoActividade
        await (0, main_1.createHistoricoActividade)(`Criou carousel`, 'Carousel', carousel.id, admin);
        carousel.imagem = await (0, aws_1.getURL)(carousel.imagem);
        res.status(201).json({ msg: 'Sucesso', carousel });
    }
    catch (err) {
        next(err);
    }
};
exports.createCarousel = createCarousel;
// Metodo que busca todas as carousels
const getCarousels = async (req, res, next) => {
    try {
        const { status } = req.query;
        const filterObj = {};
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        if (Boolean(status)) {
            filterObj.status = status;
        }
        const total = await carousel_1.default.find(filterObj).count();
        const carousels = await carousel_1.default.find(filterObj)
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage)
            .sort({ dataDespacho: -1 });
        for (const carousel of carousels) {
            if (carousel.imagem) {
                carousel.imagem = await (0, aws_1.getURL)(carousel.imagem);
            }
        }
        res.status(200).json({
            msg: 'Sucesso',
            carousels,
            ...(0, pagination_1.paginationResult)(total, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getCarousels = getCarousels;
// Metodo que busca pelo id carousel
const getCarousel = async (req, res, next) => {
    try {
        const { id } = req.params;
        const carousel = await carousel_1.default.findById(id);
        if (!carousel) {
            const error = new Error('Carousel não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (carousel.imagem) {
            carousel.imagem = await (0, aws_1.getURL)(carousel.imagem);
        }
        res.status(200).json({
            msg: 'Sucesso',
            carousel,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getCarousel = getCarousel;
// Controller para editar carousel
const editCarousel = async (req, res, next) => {
    try {
        let { carouselId, titulo, descricao, status, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const carousel = await carousel_1.default.findById(carouselId);
        if (!carousel) {
            const error = new Error('Carousel não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (Boolean(titulo)) {
            carousel.titulo = titulo;
        }
        if (Boolean(descricao)) {
            carousel.descricao = descricao;
        }
        if (Boolean(status)) {
            carousel.status = status;
            carousel.dataDespacho = new Date();
        }
        await carousel.save();
        // HistoricoActividade
        await (0, main_1.createHistoricoActividade)(`Editou carousel`, 'Carousel', carousel.id, admin);
        carousel.imagem = await (0, aws_1.getURL)(carousel.imagem);
        res.status(201).json({ msg: 'Sucesso', carousel });
    }
    catch (err) {
        next(err);
    }
};
exports.editCarousel = editCarousel;
const changeImage = async (req, res, next) => {
    try {
        const { carouselId, userId } = req.body;
        const imagem = req.file;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        if (!imagem) {
            const error = new Error('Por favor fornceça uma imagem');
            error.statusCode = 422;
            throw error;
        }
        const carousel = await carousel_1.default.findById(carouselId);
        if (!carousel) {
            const error = new Error('Carousel não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        // Primeiro deleta o arquivo
        await (0, aws_1.deleteFile)(carousel.imagem);
        // Depois renomeia e faz o upload
        imagem.filename = (0, file_1.generateFileName)(imagem);
        await (0, aws_1.uploadFile)(imagem);
        carousel.imagem = imagem.filename;
        await carousel.save();
        // HistoricoActividade
        await (0, main_1.createHistoricoActividade)(`Alterou imagem do carousel`, 'Carousel', carousel.id, admin);
        carousel.imagem = await (0, aws_1.getURL)(carousel.imagem);
        res.status(201).json({
            msg: 'Sucesso!',
            carousel,
        });
    }
    catch (error) {
        next(error);
    }
};
exports.changeImage = changeImage;
const deleteCarousel = async (req, res, next) => {
    try {
        const { id, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const user = await user_1.default.findOne({ _id: userId, isAdmin: true });
        if (!user) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const carousel = await carousel_1.default.findById(id);
        if (!carousel) {
            const error = new Error('Carousel não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        await carousel.delete();
        await (0, main_1.createHistoricoActividade)(`Deletou carousel: ${carousel.titulo}`, 'Carousel', carousel.id, user);
        res.status(204).json({ msg: 'Sucesso!' });
    }
    catch (err) {
        next(err);
    }
};
exports.deleteCarousel = deleteCarousel;
