"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deletePermante = exports.searchByNumber = exports.sendResponse = exports.changeStatus = exports.getParticipacao = exports.getParticipacoes = exports.createParticipacao = void 0;
const participacao_1 = require("../models/participacao");
const file_1 = require("../util/file");
const express_validator_1 = require("express-validator");
const participacao_2 = __importDefault(require("../models/participacao"));
const aws_1 = require("../util/aws");
const pagination_1 = require("../util/pagination");
const user_1 = __importDefault(require("../models/user"));
const main_1 = require("../util/main");
const counter_1 = __importDefault(require("../models/counter"));
const postmark_1 = __importDefault(require("../services/postmark"));
// Metodo Para criar participacao
const createParticipacao = async (req, res, next) => {
    try {
        // Anexos
        const { anexo1, anexo2 } = req.files;
        const { nome, email, tel, assunto, descricao } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        // Logica para gerar autoincremento
        const counter = await counter_1.default.findOneAndUpdate({ id: 'ParticipacaoCounter' }, { $inc: { seq: 1 } }, { new: true, upsert: true });
        const numero = counter.seq;
        const participacao = new participacao_2.default({
            nome,
            email,
            tel,
            assunto,
            descricao,
            numero,
        });
        if (anexo1) {
            anexo1[0].filename = (0, file_1.generateFileName)(anexo1[0]);
            participacao.anexo1 = anexo1[0].filename;
            await (0, aws_1.uploadFile)(anexo1[0]);
        }
        if (anexo2) {
            anexo2[0].filename = (0, file_1.generateFileName)(anexo2[0]);
            participacao.anexo2 = anexo2[0].filename;
            await (0, aws_1.uploadFile)(anexo2[0]);
        }
        await participacao.save();
        if (Boolean(email)) {
            await postmark_1.default.sendEmail({
                To: email,
                From: process.env.SENDER_EMAIL,
                Subject: `Participação nº ${participacao.numero}`,
                HtmlBody: `<h1>Participação Realisada com sucesso</h1>
        <p>
          Parabéns as sua participação foi realizada com sucesso, 
          o conselho entrará em contacto em tempo oportuno! 
        </p>
        
        <p><b>Nota:</b> Este e-mail foi gerado automaticamente pela plataforma <b>CNP</b></p>
      `,
                MessageStream: 'outbound'
            });
        }
        res.status(201).json({
            msg: 'Participação enviada com sucesso',
            participacao,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createParticipacao = createParticipacao;
// Metodo que busca todas participacoes
const getParticipacoes = async (req, res, next) => {
    try {
        const status = req.query.status;
        const filterObj = {};
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        if (Boolean(status)) {
            filterObj.status = status;
        }
        const total = await participacao_2.default.find(filterObj).count();
        const participacoes = await participacao_2.default.find(filterObj)
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage);
        for (const participacao of participacoes) {
            if (participacao.anexo1) {
                participacao.anexo1 = await (0, aws_1.getURL)(participacao === null || participacao === void 0 ? void 0 : participacao.anexo1);
            }
            if (participacao.anexo2) {
                participacao.anexo2 = await (0, aws_1.getURL)(participacao === null || participacao === void 0 ? void 0 : participacao.anexo2);
            }
        }
        res.status(200).json({
            msg: 'Sucesso',
            participacoes: participacoes,
            ...(0, pagination_1.paginationResult)(total, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getParticipacoes = getParticipacoes;
// Metodo que busca participacao pelo id
const getParticipacao = async (req, res, next) => {
    var _a, _b;
    try {
        const { id } = req.params;
        const participacao = await participacao_2.default.findById(id).populate({
            path: 'respostas.admin',
            populate: {
                path: 'dadosPessoais',
            },
        });
        if (!participacao) {
            const error = new Error('Participação não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (participacao.anexo1) {
            participacao.anexo1 = await (0, aws_1.getURL)(participacao.anexo1);
        }
        if (participacao.anexo2) {
            participacao.anexo2 = await (0, aws_1.getURL)(participacao.anexo2);
        }
        for (let resposta of participacao.respostas) {
            if ((_b = (_a = resposta.admin) === null || _a === void 0 ? void 0 : _a.dadosPessoais) === null || _b === void 0 ? void 0 : _b.fotoURL) {
                resposta.admin.dadosPessoais.fotoURL = await (0, aws_1.getURL)(resposta.admin.dadosPessoais.fotoURL);
            }
        }
        res.status(200).json({ msg: 'Sucesso', participacao: participacao });
    }
    catch (err) {
        next(err);
    }
};
exports.getParticipacao = getParticipacao;
// Metodo para mudar status
const changeStatus = async (req, res, next) => {
    var _a, _b;
    try {
        const { id, status, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        if (!participacao_1.participacaoStatus.includes(status)) {
            const error = new Error('Status inválido!');
            error.statusCode = 422;
            throw error;
        }
        const user = await user_1.default.findById(userId);
        if (!user) {
            const error = new Error('Utilizador não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const participacao = await participacao_2.default.findById(id);
        if (!participacao) {
            const error = new Error('Participação não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        participacao.status = status;
        participacao.dataDespacho = new Date().toISOString();
        await participacao.save();
        if (participacao.anexo1) {
            participacao.anexo1 = await (0, aws_1.getURL)(participacao.anexo1);
        }
        if (participacao.anexo2) {
            participacao.anexo2 = await (0, aws_1.getURL)(participacao.anexo2);
        }
        for (let resposta of participacao.respostas) {
            if ((_b = (_a = resposta.admin) === null || _a === void 0 ? void 0 : _a.dadosPessoais) === null || _b === void 0 ? void 0 : _b.fotoURL) {
                resposta.admin.dadosPessoais.fotoURL = await (0, aws_1.getURL)(resposta.admin.dadosPessoais.fotoURL);
            }
        }
        res.status(201).json({ msg: 'Sucesso', participacao: participacao });
    }
    catch (err) {
        next(err);
    }
};
exports.changeStatus = changeStatus;
// Metodo para enviar resposta
const sendResponse = async (req, res, next) => {
    try {
        const { id, userId, content } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const participacao = await participacao_2.default.findById(id);
        if (!participacao) {
            const error = new Error('Participação não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        participacao.respostas.push({
            admin: userId,
            content: content,
            data: new Date(),
        });
        await participacao.save();
        if (participacao.anexo1) {
            participacao.anexo1 = await (0, aws_1.getURL)(participacao.anexo1);
        }
        if (participacao.anexo2) {
            participacao.anexo2 = await (0, aws_1.getURL)(participacao.anexo2);
        }
        await (0, main_1.createHistoricoActividade)('Respondeu participação', 'Participacao', participacao.id, admin);
        await postmark_1.default.sendEmail({
            To: participacao.email,
            From: process.env.SENDER_EMAIL,
            Subject: `Participação`,
            HtmlBody: `<h1>Resposta à Participação Número ${participacao.numero}:</h1>
      <p style="font-size: 12pt">${participacao.respostas[participacao.respostas.length - 1].content}</p>
          `,
            MessageStream: 'outbound'
        });
        res.status(201).json({ msg: 'Sucesso', participacao: participacao });
    }
    catch (err) {
        next(err);
    }
};
exports.sendResponse = sendResponse;
// Metodo que busca participacao pelo numero
const searchByNumber = async (req, res, next) => {
    try {
        const { numero } = req.body;
        const participacao = await participacao_2.default.findOne({
            numero: numero,
        }).populate({
            path: 'respostas.admin',
            populate: {
                path: 'dadosPessoais',
            },
        });
        if (!participacao) {
            const error = new Error('Participação não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        if (participacao.anexo1) {
            participacao.anexo1 = await (0, aws_1.getURL)(participacao.anexo1);
        }
        if (participacao.anexo2) {
            participacao.anexo2 = await (0, aws_1.getURL)(participacao.anexo2);
        }
        res.status(200).json({ msg: 'Sucesso', participacao: participacao });
    }
    catch (err) {
        next(err);
    }
};
exports.searchByNumber = searchByNumber;
const deletePermante = async (req, res, next) => {
    try {
        const { participacaoId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const participacao = await participacao_2.default.findById(participacaoId);
        if (!participacao) {
            const error = new Error(`Participação não encontrada!`);
            error.statusCode = 422;
            throw error;
        }
        participacao.anexo1 && (await (0, aws_1.deleteFile)(participacao.anexo1));
        participacao.anexo2 && (await (0, aws_1.deleteFile)(participacao.anexo2));
        await participacao_2.default.deleteOne({ _id: participacaoId });
        res.status(204).json({
            msg: 'Deletado com sucesso',
        });
    }
    catch (error) {
        next(error);
    }
};
exports.deletePermante = deletePermante;
