"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.removerDuplicados = exports.getSF = exports.getSFsByMembro = exports.getSFs = exports.createSituacaoFinanceira = void 0;
const situacao_financeira_1 = __importDefault(require("../models/situacao-financeira"));
const express_validator_1 = require("express-validator");
const membro_1 = __importDefault(require("../models/membro"));
const pagination_1 = require("../util/pagination");
const aws_1 = require("../util/aws");
const pagamento_1 = __importDefault(require("../models/pagamento"));
const main_1 = require("../util/main");
const user_1 = __importDefault(require("../models/user"));
/*
  Esta controller cria SituacaoFinaceira em um ano.
  Possui um amonth inicial e um final optional, este
  se nao providenciado apenas criar a situacao para o
  mes initial
*/
const createSituacaoFinanceira = async (req, res, next) => {
    try {
        let { ano, mesInicial, mesFinal, membroId, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        const sFinaceiras = [];
        let termino = mesInicial;
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        if (Boolean(mesFinal)) {
            termino = mesFinal;
            if (mesInicial > mesFinal) {
                const error = new Error(`Mês inicial maior que final!`);
                error.statusCode = 422;
                throw error;
            }
        }
        if (termino > 12) {
            const error = new Error(`Mês inválido: ${termino}`);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado');
            error.statusCode = 422;
            throw error;
        }
        const membro = await membro_1.default.findById(membroId);
        if (!membro) {
            const error = new Error('Membro não encontrado');
            error.statusCode = 422;
            throw error;
        }
        for (let mes = mesInicial; mes <= termino; mes++) {
            let situacaoFinanceira = await situacao_financeira_1.default.findOne({
                ano: ano,
                mes: mes,
                membro: membroId,
            });
            if (!situacaoFinanceira) {
                situacaoFinanceira = await situacao_financeira_1.default.create({
                    ano: ano,
                    mes: mes,
                    membro: membroId,
                    admin: admin.id,
                });
                if (admin) {
                    let action = `Criou situação financeira do membro ${membro.id}`;
                    await (0, main_1.createHistoricoActividade)(action, 'SituacaoFinanceira', situacaoFinanceira.id, admin);
                }
            }
            sFinaceiras.push(situacaoFinanceira);
        }
        res.status(201).json({ msg: 'Sucesso', sFinaceiras });
    }
    catch (err) {
        next(err);
    }
};
exports.createSituacaoFinanceira = createSituacaoFinanceira;
// Controller Para pegar todas as SituacaoFinanceiras
const getSFs = async (req, res, next) => {
    try {
        const { status } = req.query;
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        const filterObject = {};
        if (Boolean(status)) {
            filterObject.status = status;
        }
        const total = await situacao_financeira_1.default.find(filterObject).count();
        const situacaoFinanceiras = await situacao_financeira_1.default.find(filterObject)
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage)
            .populate({
            path: 'membro',
            populate: [{ path: 'dadosPessoais' }],
        })
            .sort({ ano: 1, mes: 1 });
        res.status(200).json({
            msg: 'Sucesso',
            situacaoFinanceiras,
            ...(0, pagination_1.paginationResult)(total, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getSFs = getSFs;
// Controller Para pegar todas as SituacaoFinanceiras de um membro
const getSFsByMembro = async (req, res, next) => {
    try {
        const { membroId } = req.params;
        const { status, ano } = req.query;
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 48;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        const filterObject = {
            membro: membroId,
        };
        if (Boolean(status)) {
            filterObject.status = status;
        }
        if (Boolean(ano)) {
            filterObject.ano = ano;
        }
        const total = await situacao_financeira_1.default.find(filterObject).count();
        const situacaoFinanceiras = await situacao_financeira_1.default.find(filterObject)
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage)
            .populate({
            path: 'membro',
            populate: [{ path: 'dadosPessoais' }],
        })
            .sort({ ano: -1, mes: 1 });
        res.status(200).json({
            msg: 'Sucesso',
            situacaoFinanceiras,
            ...(0, pagination_1.paginationResult)(total, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getSFsByMembro = getSFsByMembro;
// Controller que pega um único SituacaoFinanceira
const getSF = async (req, res, next) => {
    try {
        const { id } = req.params;
        const situacaoFinanceira = await situacao_financeira_1.default.findById(id).populate({
            path: 'membro',
            populate: [{ path: 'dadosPessoais' }],
        });
        if (!situacaoFinanceira) {
            const error = new Error('Situação financeira não encontrada!');
            error.statusCode = 422;
            throw error;
        }
        const membro = situacaoFinanceira.membro;
        const pagamento = await pagamento_1.default.findOne({
            ano: situacaoFinanceira.ano,
            mes: situacaoFinanceira.mes,
            dadosPessoais: membro.dadosPessoais.id,
        })
            .populate('referencia')
            .populate('emolumento')
            .populate('aprovadoPor');
        if (pagamento === null || pagamento === void 0 ? void 0 : pagamento.comprovativo) {
            pagamento.comprovativo = await (0, aws_1.getURL)(pagamento.comprovativo);
        }
        res.status(200).json({ msg: 'Sucesso', situacaoFinanceira, pagamento });
    }
    catch (err) {
        next(err);
    }
};
exports.getSF = getSF;
const removerDuplicados = async (req, res, next) => {
    try {
        // Agrupa os registros do ano 2025 por mês e membro
        const duplicados = await situacao_financeira_1.default.aggregate([
            { $match: { ano: 2025 } }, // Filtra registros do ano 2025
            {
                $group: {
                    _id: { mes: "$mes", membro: "$membro" }, // Agrupa por mês e membro
                    ids: { $push: "$_id" }, // Coleta os IDs duplicados
                    maisRecente: { $max: "$updatedAt" } // Identifica o mais recente
                }
            }
        ]);
        let removidos = 0;
        for (const doc of duplicados) {
            if (doc.ids.length > 1) {
                // Mantém apenas um e remove os outros
                const idsParaRemover = doc.ids.slice(1);
                const result = await situacao_financeira_1.default.deleteMany({
                    _id: { $in: idsParaRemover }
                });
                removidos += result.deletedCount;
            }
            console.log(removidos);
        }
        return res.status(200).json({
            message: `Duplicatas removidas com sucesso! Registros excluídos: ${removidos}`,
        });
    }
    catch (error) {
        console.error("Erro ao remover duplicatas:", error);
        // return res.status(500).json({
        //     message: "Erro interno ao remover duplicatas",
        //     error: error.message
        // });
        next(error);
    }
};
exports.removerDuplicados = removerDuplicados;
