"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteArtigo = exports.searchArtigos = exports.editArtigo = exports.getArtigo = exports.searchArtigoByNumero = exports.getArtigosByCapitulo = exports.getArtigos = exports.registerArtigo = void 0;
const artigo_1 = __importDefault(require("../models/artigo"));
const express_validator_1 = require("express-validator");
const capitulo_1 = __importDefault(require("../models/capitulo"));
const user_1 = __importDefault(require("../models/user"));
const main_1 = require("../util/main");
// Funcao para registar artigo
const registerArtigo = async (req, res, next) => {
    try {
        let { titulo, numero, alineas, capituloId, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const foundArtigo = await artigo_1.default.findOne({ numero: numero, capitulo: capituloId })
            .populate('capitulo');
        if (foundArtigo) {
            const capitulo = foundArtigo.capitulo;
            const error = new Error(`Artigo ${numero}º do capítulo nº ${capitulo.numero} já foi inserido!`);
            error.statusCode = 422;
            throw error;
        }
        const artigo = new artigo_1.default({
            titulo,
            numero,
            alineas,
            capitulo: capituloId
        });
        await artigo.save();
        await (0, main_1.createHistoricoActividade)('Inseriu Artigo', 'Artigo', artigo.id, admin);
        res.status(201).json({ msg: 'Sucesso', artigo });
    }
    catch (err) {
        next(err);
    }
};
exports.registerArtigo = registerArtigo;
// Funcao Para pegar todas as artigos
const getArtigos = async (req, res, next) => {
    try {
        const artigos = await artigo_1.default.find()
            .populate('capitulo')
            .sort({ numero: 1 });
        res.status(200).json({ msg: "Sucesso", artigos });
    }
    catch (err) {
        next(err);
    }
};
exports.getArtigos = getArtigos;
// Funcao Para pegar todas as artigos
const getArtigosByCapitulo = async (req, res, next) => {
    try {
        const { capituloId } = req.params;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const capitulo = await capitulo_1.default.findById(capituloId);
        if (!capitulo) {
            const error = new Error('Capítulo não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const artigos = await artigo_1.default.find({ capitulo: capituloId })
            .populate('capitulo')
            .sort({ numero: 1 });
        res.status(200).json({ msg: "Sucesso", artigos });
    }
    catch (err) {
        next(err);
    }
};
exports.getArtigosByCapitulo = getArtigosByCapitulo;
// Funcao para pesquisar artigo pelo numero
const searchArtigoByNumero = async (req, res, next) => {
    try {
        let { numero } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const artigo = await artigo_1.default.findOne({ numero: numero })
            .populate('capitulo');
        if (!artigo) {
            const error = new Error('Artigo não encontrado');
            error.statusCode = 422;
            throw error;
        }
        res.status(200).json({ msg: "Sucesso", artigo });
    }
    catch (err) {
        next(err);
    }
};
exports.searchArtigoByNumero = searchArtigoByNumero;
// Funcao que pega uma artigo
const getArtigo = async (req, res, next) => {
    try {
        const { id } = req.params;
        const artigo = await artigo_1.default.findById(id)
            .populate('capitulo');
        if (!artigo) {
            const error = new Error('Artigo não encontrada');
            error.statusCode = 422;
            throw error;
        }
        res.status(200).json({ msg: "Sucesso", artigo });
    }
    catch (err) {
        next(err);
    }
};
exports.getArtigo = getArtigo;
// Metodo que actualiza artigo
const editArtigo = async (req, res, next) => {
    try {
        const { artigoId, titulo, capituloId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const artigo = await artigo_1.default.findById(artigoId);
        if (!artigo) {
            const error = new Error('Artigo não encontrada');
            error.statusCode = 422;
            throw error;
        }
        artigo.titulo = titulo;
        artigo.capitulo = capituloId;
        await artigo.save();
        res.status(201).json({ msg: "Sucesso", artigo });
    }
    catch (err) {
        next(err);
    }
};
exports.editArtigo = editArtigo;
// Funcao para pesquisar artigo pelo titulo
const searchArtigos = async (req, res, next) => {
    try {
        const { titulo } = req.body;
        const regex = new RegExp(titulo, "i");
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const artigos = await artigo_1.default.find({ titulo: regex });
        res.status(200).json({ msg: "Sucesso", artigos });
    }
    catch (err) {
        next(err);
    }
};
exports.searchArtigos = searchArtigos;
// Metodo para deletar artigo
const deleteArtigo = async (req, res, next) => {
    try {
        const { artigoId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const artigo = await artigo_1.default.findById(artigoId);
        if (!artigo) {
            const error = new Error('Artigo não encontrada');
            error.statusCode = 422;
            throw error;
        }
        await artigo.delete();
        res.status(204).json({ msg: "Sucesso", artigo });
    }
    catch (err) {
        next(err);
    }
};
exports.deleteArtigo = deleteArtigo;
