"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deleteCapitulo = exports.searchCapitulos = exports.editCapitulo = exports.getCapitulo = exports.searchCapituloByNumero = exports.getCapitulos = exports.registerCapitulo = void 0;
const capitulo_1 = __importDefault(require("../models/capitulo"));
const express_validator_1 = require("express-validator");
const user_1 = __importDefault(require("../models/user"));
const main_1 = require("../util/main");
// Funcao Para registar capitulo
const registerCapitulo = async (req, res, next) => {
    try {
        let { titulo, numero, userId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const admin = await user_1.default.findById(userId);
        if (!admin) {
            const error = new Error('Admin não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        const foundCapitulo = await capitulo_1.default.findOne({ numero: numero });
        if (foundCapitulo) {
            const error = new Error(`Capítulo nº ${numero} já foi inserido!`);
            error.statusCode = 422;
            throw error;
        }
        const capitulo = new capitulo_1.default({
            titulo,
            numero
        });
        await capitulo.save();
        await (0, main_1.createHistoricoActividade)('Inseriu Capítulo', 'Capitulo', capitulo.id, admin);
        res.status(201).json({ msg: 'Sucesso', capitulo });
    }
    catch (err) {
        next(err);
    }
};
exports.registerCapitulo = registerCapitulo;
// Funcao Para pegar todas as capitulos
const getCapitulos = async (req, res, next) => {
    try {
        const capitulos = await capitulo_1.default.find()
            .sort({ numero: 1 });
        res.status(200).json({ msg: "Sucesso", capitulos });
    }
    catch (err) {
        next(err);
    }
};
exports.getCapitulos = getCapitulos;
// Funcao para pesquisar capitulo pelo numero
const searchCapituloByNumero = async (req, res, next) => {
    try {
        let { numero } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const capitulo = await capitulo_1.default.findOne({ numero: numero });
        if (!capitulo) {
            const error = new Error('Capítulo não encontrado');
            error.statusCode = 422;
            throw error;
        }
        res.status(200).json({ msg: "Sucesso", capitulo });
    }
    catch (err) {
        next(err);
    }
};
exports.searchCapituloByNumero = searchCapituloByNumero;
// Funcao que pega uma capitulo
const getCapitulo = async (req, res, next) => {
    try {
        const { id } = req.params;
        const capitulo = await capitulo_1.default.findById(id);
        if (!capitulo) {
            const error = new Error('Capítulo não encontrado');
            error.statusCode = 422;
            throw error;
        }
        res.status(200).json({ msg: "Sucesso", capitulo: capitulo });
    }
    catch (err) {
        next(err);
    }
};
exports.getCapitulo = getCapitulo;
// Metodo que actualiza capitulo
const editCapitulo = async (req, res, next) => {
    try {
        const { capituloId, titulo } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const capitulo = await capitulo_1.default.findById(capituloId);
        if (!capitulo) {
            const error = new Error('Capítulo não encontrado');
            error.statusCode = 422;
            throw error;
        }
        capitulo.titulo = titulo;
        await capitulo.save();
        res.status(201).json({ msg: "Sucesso", capitulo });
    }
    catch (err) {
        next(err);
    }
};
exports.editCapitulo = editCapitulo;
// Funcao para pesquisar capitulo pelo titulo
const searchCapitulos = async (req, res, next) => {
    try {
        const { titulo } = req.body;
        const regex = new RegExp(titulo, "i");
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const capitulos = await capitulo_1.default.find({ titulo: regex });
        res.status(200).json({ msg: "Sucesso", capitulos });
    }
    catch (err) {
        next(err);
    }
};
exports.searchCapitulos = searchCapitulos;
// Metodo para deletar capitulo
const deleteCapitulo = async (req, res, next) => {
    try {
        const { capituloId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const capitulo = await capitulo_1.default.findById(capituloId);
        if (!capitulo) {
            const error = new Error('Capítulo não encontrado');
            error.statusCode = 422;
            throw error;
        }
        await capitulo.delete();
        res.status(204).json({ msg: "Sucesso", capitulo });
    }
    catch (err) {
        next(err);
    }
};
exports.deleteCapitulo = deleteCapitulo;
