"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.viewMessage = exports.getMembrosChat = exports.getMembroAdminMessages = exports.sendMessageMembroToAdmin = exports.sendMessageAdminToMembro = void 0;
const dados_pessoais_1 = __importDefault(require("../models/dados-pessoais"));
const message_1 = __importDefault(require("../models/message"));
const express_validator_1 = require("express-validator");
const aws_1 = require("../util/aws");
const pagination_1 = require("../util/pagination");
const validate_url_1 = __importDefault(require("../util/validate-url"));
const chat_1 = __importDefault(require("../models/chat"));
const membro_1 = __importDefault(require("../models/membro"));
const main_1 = require("../util/main");
const user_1 = __importDefault(require("../models/user"));
// Controller para enviar mensagem de Admin para membro
const sendMessageAdminToMembro = async (req, res, next) => {
    try {
        let { sender, receiver, content } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        let chat;
        // Variavél para determinar se é um chat com o admin
        const withAdmin = true;
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        if (sender === receiver) {
            const error = new Error('Dados pessoais iguais!');
            error.statusCode = 422;
            throw error;
        }
        const dadosSender = await dados_pessoais_1.default.findById(sender);
        if (!dadosSender) {
            const error = new Error('Dados pessoais de sender não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        // Verifica se o perfil do sender é de admin
        if (dadosSender.perfil === 'Farmacêutico(a)') {
            const error = new Error(`Perfil do sender inválido '${dadosSender.perfil}'`);
            error.statusCode = 422;
            throw error;
        }
        const dadosReceiver = await dados_pessoais_1.default.findById(receiver);
        if (!dadosReceiver) {
            const error = new Error('Dados pessoais de receiver não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        // Verifica se o perfil do receiver é de membro
        if (dadosReceiver.perfil !== 'Farmacêutico(a)' &&
            dadosReceiver.perfil !== 'Presidente' &&
            dadosReceiver.perfil !== 'Vice-Presidente' &&
            dadosReceiver.perfil !== 'CNP' // Para Teste
        ) {
            const error = new Error(`Perfil do receiver inválido '${dadosReceiver.perfil}'`);
            error.statusCode = 422;
            throw error;
        }
        chat = await chat_1.default.findOne({ dadosMembro: receiver });
        if (!chat) {
            chat = await chat_1.default.create({ dadosMembro: receiver });
        }
        const msg = new message_1.default({
            sender,
            receiver,
            content,
            withAdmin,
            chat: chat.id,
        });
        await msg.save();
        // getIO().emit('message', { action: 'sent', mensagem: msg });
        const admin = await user_1.default.findOne({ dadosPessoais: sender });
        if (admin) {
            await (0, main_1.createHistoricoActividade)('Respondeu ao chat', 'Chat', chat.id, admin);
        }
        res.status(201).json({ msg: 'Sucesso', mensagem: msg });
    }
    catch (err) {
        next(err);
    }
};
exports.sendMessageAdminToMembro = sendMessageAdminToMembro;
// Controller para o membro enviar mensagem para admin
const sendMessageMembroToAdmin = async (req, res, next) => {
    try {
        let { sender, content } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        let chat;
        // Variavél para determinar se é um chat com o admin
        const withAdmin = true;
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const dadosSender = await dados_pessoais_1.default.findById(sender);
        if (!dadosSender) {
            const error = new Error('Dados pessoais de sender não encontrado!');
            error.statusCode = 422;
            throw error;
        }
        if (dadosSender.perfil !== 'Farmacêutico(a)' &&
            dadosSender.perfil !== 'Presidente' &&
            dadosSender.perfil !== 'Vice-Presidente' &&
            dadosSender.perfil !== 'CNP' // Para teste
        ) {
            const error = new Error(`Perfil inválido '${dadosSender.perfil}'`);
            error.statusCode = 422;
            throw error;
        }
        chat = await chat_1.default.findOne({ dadosMembro: sender });
        if (!chat) {
            chat = await chat_1.default.create({ dadosMembro: sender });
        }
        const msg = new message_1.default({
            sender,
            content,
            withAdmin,
            chat: chat.id,
        });
        await msg.save();
        // getIO().emit('message', { action: 'sent', mensagem: msg });
        res.status(201).json({ msg: 'Sucesso', mensagem: msg });
    }
    catch (err) {
        next(err);
    }
};
exports.sendMessageMembroToAdmin = sendMessageMembroToAdmin;
// Controller para pegar as mensagens entre membros e admins
const getMembroAdminMessages = async (req, res, next) => {
    var _a, _b, _c, _d;
    try {
        const { dadosPessoaisId } = req.params;
        const filterObj = {
            $and: [
                { withAdmin: true },
                { $or: [{ sender: dadosPessoaisId }, { receiver: dadosPessoaisId }] },
            ],
        };
        let pageQuery = req.query.page;
        let page = (0, pagination_1.getPage)(pageQuery);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        const dadosPessoais = await dados_pessoais_1.default.findById(dadosPessoaisId);
        if (!dadosPessoais) {
            const error = new Error(`Dados pessoais não encontrado!`);
            error.statusCode = 422;
            throw error;
        }
        const total = await message_1.default.find(filterObj).count();
        // Pega a ultima pagina
        if (!pageQuery) {
            page = (0, pagination_1.paginationResult)(total, page, perPage).lastPage;
        }
        let skipValue = (0, pagination_1.skip)(page, perPage);
        if (page === 0) {
            skipValue = 0;
        }
        const messages = await message_1.default.find(filterObj)
            .populate(['sender', 'receiver'])
            .skip(skipValue)
            .limit(perPage);
        for (const message of messages) {
            if ((message === null || message === void 0 ? void 0 : message.sender) && !(0, validate_url_1.default)((_a = message === null || message === void 0 ? void 0 : message.sender) === null || _a === void 0 ? void 0 : _a.fotoURL)) {
                message.sender.fotoURL = await (0, aws_1.getURL)((_b = message === null || message === void 0 ? void 0 : message.sender) === null || _b === void 0 ? void 0 : _b.fotoURL);
            }
            if ((message === null || message === void 0 ? void 0 : message.receiver) && !(0, validate_url_1.default)((_c = message === null || message === void 0 ? void 0 : message.receiver) === null || _c === void 0 ? void 0 : _c.fotoURL)) {
                message.receiver.fotoURL = await (0, aws_1.getURL)((_d = message === null || message === void 0 ? void 0 : message.receiver) === null || _d === void 0 ? void 0 : _d.fotoURL);
            }
        }
        res.status(200).json({
            msg: 'Sucesso',
            messages,
            ...(0, pagination_1.paginationResult)(total, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getMembroAdminMessages = getMembroAdminMessages;
// Controller para pegar todos os chats
const getMembrosChat = async (req, res, next) => {
    try {
        const status = req.query.status;
        const filterObj = {};
        let membrosMgs = 0;
        const page = (0, pagination_1.getPage)(req.query.page);
        let paramPerPage = req.query.perPage;
        let perPage = 50;
        if (paramPerPage) {
            perPage = +paramPerPage;
        }
        if (filterObj) {
            filterObj.status = status;
        }
        const total = await message_1.default.aggregate([
            {
                $group: {
                    _id: { chat: '$chat' },
                    messages: { $push: '$$ROOT' },
                },
            },
            {
                $project: {
                    _id: 0,
                    sender: '$_id.sender',
                    receiver: '$_id.receiver',
                    messages: {
                        $slice: ['$messages', -1], // Obter apenas a última mensagem do array (a mais recente)
                    },
                },
            },
            {
                $sort: {
                    'messages.createdAt': -1, // Ordenar as mensagens pela data mais recente (decrescente)
                },
            },
        ]).count('messages');
        let chats = await message_1.default.aggregate([
            {
                $group: {
                    _id: { chat: '$chat' },
                    messages: { $push: '$$ROOT' },
                },
            },
            {
                $project: {
                    _id: 0,
                    sender: '$_id.sender',
                    receiver: '$_id.receiver',
                    messages: {
                        $slice: ['$messages', -1], // Obter apenas a última mensagem do array (a mais recente)
                    },
                },
            },
            {
                $sort: {
                    'messages.createdAt': -1, // Ordenar as mensagens pela data mais recente (decrescente)
                },
            },
            {
                $lookup: {
                    from: 'chats', // Nome da coleção do modelo Chat
                    localField: 'messages.chat',
                    foreignField: '_id',
                    as: 'chatDetails',
                },
            },
            {
                $unwind: '$chatDetails', // Desconstruir o array do $lookup (opcional, se houver apenas um remetente por grupo)
            },
            {
                $lookup: {
                    from: 'dadospessoais', // Nome da coleção do modelo DadosPessoais
                    localField: 'messages.sender',
                    foreignField: '_id',
                    as: 'senderDetails',
                },
            },
            {
                $unwind: '$senderDetails',
            },
            {
                $project: {
                    sender: 1,
                    receiver: 1,
                    messages: {
                        $map: {
                            input: '$messages',
                            as: 'message',
                            in: {
                                $mergeObjects: [
                                    '$$message',
                                    {
                                        chatDetails: {
                                            _id: '$chatDetails._id',
                                            dadosMembro: '$chatDetails.dadosMembro', // Adicione outros campos do usuário, se necessário
                                        },
                                        senderDetails: {
                                            _id: '$senderDetails._id',
                                            nome: '$senderDetails.nome',
                                            perfil: '$senderDetails.perfil',
                                            genero: '$senderDetails.genero',
                                        },
                                    },
                                ],
                            },
                        },
                    },
                },
            },
        ])
            .skip((0, pagination_1.skip)(page, perPage))
            .limit(perPage);
        for (const obj of chats) {
            const chatDetails = obj.messages[0].chatDetails;
            const senderDetails = obj.messages[0].senderDetails;
            const membro = await membro_1.default.findOne({
                dadosPessoais: chatDetails.dadosMembro,
            }).populate('dadosPessoais');
            const dadosPessoais = membro === null || membro === void 0 ? void 0 : membro.dadosPessoais;
            if (dadosPessoais && !(0, validate_url_1.default)(dadosPessoais === null || dadosPessoais === void 0 ? void 0 : dadosPessoais.fotoURL)) {
                dadosPessoais.fotoURL = await (0, aws_1.getURL)(dadosPessoais === null || dadosPessoais === void 0 ? void 0 : dadosPessoais.fotoURL);
                chatDetails.dadosMembro = membro;
            }
            if (senderDetails._id !== (dadosPessoais === null || dadosPessoais === void 0 ? void 0 : dadosPessoais._id) &&
                (senderDetails.perfil === 'Farmacêutico(a)' ||
                    senderDetails.perfil === 'Vice-Presidente' ||
                    senderDetails.perfil === 'Presidente')) {
                if (membro && !obj.messages[0].viewed) {
                    senderDetails.nome = membro === null || membro === void 0 ? void 0 : membro.nome;
                    membrosMgs++;
                }
            }
        }
        res.status(200).json({
            msg: 'Sucesso',
            membrosMgs,
            chats,
            ...(0, pagination_1.paginationResult)(total[0].messages, page, perPage),
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getMembrosChat = getMembrosChat;
// Controller para messagens uma messangem como lida
const viewMessage = async (req, res, next) => {
    var _a, _b, _c, _d;
    try {
        const { messageId } = req.body;
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const message = await message_1.default.findById(messageId).populate([
            'sender',
            'receiver',
        ]);
        if (message) {
            if ((message === null || message === void 0 ? void 0 : message.sender) && !(0, validate_url_1.default)((_a = message === null || message === void 0 ? void 0 : message.sender) === null || _a === void 0 ? void 0 : _a.fotoURL)) {
                message.sender.fotoURL = await (0, aws_1.getURL)((_b = message === null || message === void 0 ? void 0 : message.sender) === null || _b === void 0 ? void 0 : _b.fotoURL);
            }
            if ((message === null || message === void 0 ? void 0 : message.receiver) && !(0, validate_url_1.default)((_c = message === null || message === void 0 ? void 0 : message.receiver) === null || _c === void 0 ? void 0 : _c.fotoURL)) {
                message.receiver.fotoURL = await (0, aws_1.getURL)((_d = message === null || message === void 0 ? void 0 : message.receiver) === null || _d === void 0 ? void 0 : _d.fotoURL);
            }
            message.viewed = !message.viewed;
            await message.save();
        }
        res.status(200).json({
            msg: 'Sucesso',
            message,
        });
    }
    catch (err) {
        next(err);
    }
};
exports.viewMessage = viewMessage;
