"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.deletePergunta = exports.getOnePergunta = exports.getAllPerguntas = exports.createPergunta = void 0;
const express_validator_1 = require("express-validator");
const pergunta_1 = __importDefault(require("../models/pergunta"));
const mongoose_1 = __importDefault(require("mongoose"));
const inquerito_1 = __importDefault(require("../models/inquerito"));
const createPergunta = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const { texto, inquerito } = req.body;
        const verifyInquerito = await inquerito_1.default.findOne({ _id: inquerito });
        if (!verifyInquerito) {
            const error = new Error(`Este inquérito não existe!`);
            error.statusCode = 404;
            throw error;
        }
        const verifyPergunta = await pergunta_1.default.findOne({ texto, inquerito });
        if (verifyPergunta) {
            const error = new Error(`O Pergunta: (${verifyPergunta.texto}) já foi criado para este inquérito! `);
            error.statusCode = 409;
            throw error;
        }
        const pergunta = new pergunta_1.default({
            texto,
            inquerito
        });
        // Adicionar a pergunta ao array perguntas do Inquérito se não estiver presente
        if (!verifyInquerito.perguntas.includes(pergunta._id)) {
            verifyInquerito.perguntas.push(pergunta._id);
            await verifyInquerito.save(); // Salvar as mudanças no Evento
        }
        await pergunta.save();
        res.status(201).json({
            msg: 'Pergunta criado com sucesso',
            pergunta
        });
    }
    catch (err) {
        next(err);
    }
};
exports.createPergunta = createPergunta;
const getAllPerguntas = async (req, res, next) => {
    try {
        const perguntas = await pergunta_1.default.find({});
        if (perguntas.length === 0) {
            return res.status(404).json({
                success: false,
                perguntas
            });
        }
        res.status(200).json({
            success: true,
            perguntas
        });
    }
    catch (err) {
        next(err);
    }
};
exports.getAllPerguntas = getAllPerguntas;
const getOnePergunta = async (req, res, next) => {
    try {
        const id = req.params.id;
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (validId) {
            const pergunta = await pergunta_1.default.findOne({ _id: id });
            if (!pergunta) {
                return res.status(404).json({
                    success: false,
                    pergunta
                });
            }
            res.status(200).json({
                success: true,
                pergunta
            });
        }
        else {
            return res.status(400).json({
                success: false,
                message: 'Id inválido.'
            });
        }
    }
    catch (err) {
        next(err);
    }
};
exports.getOnePergunta = getOnePergunta;
const deletePergunta = async (req, res, next) => {
    try {
        const errors = (0, express_validator_1.validationResult)(req);
        if (!errors.isEmpty()) {
            const error = new Error(errors.array()[0].msg);
            error.statusCode = 422;
            throw error;
        }
        const id = req.params.id;
        // Verifica se o id é válido
        const validId = mongoose_1.default.Types.ObjectId.isValid(id) ? id : null;
        if (!validId) {
            return res.status(400).json({
                success: false,
                message: 'ID inválido.',
            });
        }
        // Busca a Pergunta para garantir que ele existe
        const pergunta = await pergunta_1.default.findById(id);
        if (!pergunta) {
            return res.status(404).json({
                success: false,
                message: 'Pergunta não encontrada.',
            });
        }
        // Busca e atualiza todos os inquéritos que contêm esta pergunta
        await inquerito_1.default.updateMany({ perguntas: pergunta._id }, { $pull: { perguntas: pergunta._id } });
        await pergunta_1.default.findByIdAndDelete(id);
        res.status(200).json({
            success: true,
            message: `Pergunta deletada com sucesso.`,
        });
    }
    catch (err) {
        next(err); // Passa o erro para o middleware de erro
    }
};
exports.deletePergunta = deletePergunta;
