"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.addFecho = exports.checkReferences = exports.createAnnuallySF = exports.checkMonthlySF = exports.createSF = exports.sendPaymentNotification = exports.checkMembroSF = exports.checkAllMembrosSF = exports.createManySF = exports.postProxyPay = exports.generateReferenceIds = exports.generateReference = exports.callBackURL = void 0;
const situacao_financeira_1 = __importDefault(require("../models/situacao-financeira"));
const membro_1 = __importDefault(require("../models/membro"));
const main_1 = require("./main");
const referencia_1 = __importDefault(require("../models/referencia"));
const pagamento_1 = __importDefault(require("../models/pagamento"));
const proxypay_1 = __importDefault(require("../services/proxypay"));
const pagamento_referencia_1 = __importDefault(require("../models/pagamento-referencia"));
const fecho_1 = __importDefault(require("../models/fecho"));
const postmark_1 = __importDefault(require("../services/postmark"));
let runningCheckReference;
exports.callBackURL = `https://ofa-production-server.vamsolucoes.com/api/v2/payments/proxypay`;
// Gera referencia atraves dos API da ProxyPay
const generateReference = async (emolumento, dadosPessoais, dataExpiracao, costumFields, callBackEndPoint, admin) => {
    const ref = await (0, exports.generateReferenceIds)();
    const callback_url = callBackEndPoint ? exports.callBackURL + callBackEndPoint : exports.callBackURL;
    await proxypay_1.default.put(`references/${ref}`, {
        amount: emolumento.valor,
        end_datetime: dataExpiracao,
        custom_fields: {
            emolumentoId: emolumento.id,
            dadosPessoaisId: dadosPessoais.id,
            callback_url
        },
    });
    const referencia = new referencia_1.default({
        montante: emolumento.valor,
        ref: ref,
        dadosPessoais,
        emolumento: emolumento,
        dataExpiracao: dataExpiracao,
        entidade: 1067,
    });
    const pagamento = new pagamento_1.default({
        valorPagar: emolumento.valor,
        ref: ref,
        referencia,
        dadosPessoais: dadosPessoais,
        emolumento: emolumento,
        entidade: 1067,
    });
    if (admin) {
        referencia.admin = admin;
        pagamento.admin = admin;
    }
    if (costumFields) {
        pagamento.costumFields = costumFields;
    }
    await referencia.save();
    return pagamento.save();
};
exports.generateReference = generateReference;
// Gera ReferenceIds atraves da API da ProxyPay
const generateReferenceIds = async () => {
    // Generate reference ids from ProxyPay enpoint
    let reference = await proxypay_1.default.post('reference_ids', {});
    const ref = reference.data;
    const foundReferencia = await referencia_1.default.findOne({
        ref: ref,
        status: { $ne: 'Expirada' },
    });
    const pagamentoReferencia = await pagamento_referencia_1.default.findOne({
        reference_id: ref,
    });
    if (foundReferencia || pagamentoReferencia) {
        return (0, exports.generateReferenceIds)();
    }
    return ref;
};
exports.generateReferenceIds = generateReferenceIds;
// Metodo que recebe pagamento da CallbackURL a ser postado pela ProxyPay
const postProxyPay = async (transaction_id, terminal_type, terminal_transaction_id, terminal_location, terminal_id, reference_id, product_id, period_start_datetime, period_id, period_end_datetime, parameter_id, id, fee, entity_id, datetime, custom_fields, amount) => {
    let pagamento;
    let referencia;
    let pagamentoReferencia = await pagamento_referencia_1.default.findOne({
        id: id,
    });
    if (!pagamentoReferencia) {
        pagamentoReferencia = new pagamento_referencia_1.default({
            transaction_id,
            terminal_type,
            terminal_transaction_id,
            terminal_location,
            terminal_id,
            reference_id,
            product_id,
            period_start_datetime,
            period_id,
            period_end_datetime,
            parameter_id,
            id,
            fee,
            entity_id,
            datetime,
            custom_fields,
            amount,
        });
        await pagamentoReferencia.save();
    }
    /*
      Busca a referencia e altera
      o seu estado para 'Paga'
    */
    referencia = await referencia_1.default.findOne({
        ref: reference_id,
        status: { $ne: 'Expirado' },
    });
    if (referencia) {
        referencia.status = 'Paga';
        await referencia.save();
    }
    /*
      Busca o pagamento e
      altera o seu estado para
      'Aprovado'
    */
    pagamento = await pagamento_1.default.findOne({
        ref: reference_id,
    }).populate('emolumento');
    if (!pagamento) {
        await pagamento_1.default.create({
            valorPago: amount,
            ref: reference_id,
            referencia,
            pagamentoReferencia: pagamentoReferencia,
            status: 'Aprovado',
            dataConfirmacao: new Date(datetime),
            entidade: 1067,
        });
    }
    else {
        pagamento.valorPago = amount;
        pagamento.dataConfirmacao = new Date(datetime);
        pagamento.pagamentoReferencia = pagamentoReferencia;
        pagamento.status = 'Aprovado';
        await pagamento.save();
    }
    return pagamento;
};
exports.postProxyPay = postProxyPay;
// Funcao para criar varias situacoes finaceiras
const createManySF = async (ano, mesInicio, membroId) => {
    const sFinaceiras = [];
    const membro = await membro_1.default.findById(membroId);
    if (!membro) {
        const error = new Error('Membro não encontrado!');
        error.statusCode = 422;
        throw error;
    }
    for (let mes = mesInicio; mes <= 12; mes++) {
        const foundSF = await situacao_financeira_1.default.findOne({
            ano: ano,
            mes: mes,
            membro: membroId,
        });
        if (!foundSF) {
            let situacaoFinanceira = await situacao_financeira_1.default.create({
                ano: ano,
                mes: mes,
                membro: membroId,
            });
            sFinaceiras.push(situacaoFinanceira);
        }
    }
    return sFinaceiras;
};
exports.createManySF = createManySF;
// Funcao para verificar as situacoes financeiras de todos os membros
const checkAllMembrosSF = async (mesActual, anoActual) => {
    const membros = await membro_1.default.find({
        $and: [{ status: { $ne: 'Suspenso' } }, { status: { $ne: 'Falecido' } }],
    });
    for (let membro of membros) {
        const dadosPessoais = membro.dadosPessoais;
        const countSF = await situacao_financeira_1.default.find({
            $or: [
                {
                    $and: [
                        { membro: membro.id },
                        { ano: { $lt: anoActual } },
                        { status: 'Devendo' },
                    ],
                },
                {
                    $and: [
                        { membro: membro.id },
                        { mes: { $lte: mesActual } },
                        { ano: anoActual },
                        { status: 'Devendo' },
                    ],
                },
            ],
        }).count();
        if (countSF > 0) {
            membro.status = 'Inactivo';
            await membro.save();
            await postmark_1.default.sendEmail({
                To: dadosPessoais.email,
                From: process.env.SENDER_EMAIL,
                Subject: `Inactivação da Conta`,
                HtmlBody: `<h1>Inactivação por não pagamento de cota</h1>
          <p style="font-size: 12pt">
            Informamos que o seu perfil de membro foi inactivado
            por não pagar a cota do mês de ${mesActual}.
            Pague as suas cotas e mantem-se como um membro activo.
          </p>

          <p style="font-size: 12pt">Acesse o seu perfil e faça o pagamento no link a baixo:</p>

          <p style="font-size: 12pt">Use o aplicativo CNP para fazer o pagamento</p>
          <p><i>Este e-mail foi gerado pela plataforma CNP</i></p>
        `,
                MessageStream: 'outbound',
            });
        }
        else {
            membro.status = 'Activo';
            await membro.save();
        }
    }
};
exports.checkAllMembrosSF = checkAllMembrosSF;
// Funcao para verificar as situacoes financeiras de um membro
const checkMembroSF = async (membroId, mesActual, anoActual) => {
    const membro = await membro_1.default.findOne({
        $and: [
            { id: membroId },
            { status: { $ne: 'Suspenso' } },
            { status: { $ne: 'Falecido' } },
        ],
    });
    if (membro) {
        const countSF = await situacao_financeira_1.default.find({
            $or: [
                {
                    $and: [
                        { membro: membro.id },
                        { ano: { $lt: anoActual } },
                        { status: { $ne: 'Pago' } },
                    ],
                },
                {
                    $and: [
                        { membro: membro.id },
                        { ano: anoActual },
                        { mes: { $lte: mesActual } },
                        { status: { $ne: 'Pago' } },
                    ],
                },
            ],
        }).count();
        if (countSF > 0) {
            membro.status = 'Inactivo';
            await membro.save();
        }
        else {
            membro.status = 'Activo';
            await membro.save();
        }
    }
};
exports.checkMembroSF = checkMembroSF;
// Funcao para enviar notificação de pagamento aos membros membro
const sendPaymentNotification = async () => {
    const data = new Date((0, main_1.getAngolaTime)());
    const dia = data.getDate();
    const mesActual = data.getMonth() + 1;
    const anoActual = data.getFullYear();
    const hora = data.getHours();
    const minutos = data.getMinutes();
    const segundos = data.getSeconds();
    if (dia === 20 && hora === 0 && minutos === 0 && segundos === 0) {
        const membros = await membro_1.default.find({
            $and: [{ status: { $ne: 'Suspenso' } }, { status: { $ne: 'Falecido' } }],
        });
        for (let membro of membros) {
            const dadosPessoais = membro.dadosPessoais;
            const countSF = await situacao_financeira_1.default.findOne({
                mes: mesActual,
                ano: anoActual,
                status: 'Devendo',
                membro: membro.id,
            });
            if (countSF) {
                await postmark_1.default.sendEmail({
                    To: dadosPessoais.email,
                    From: process.env.SENDER_EMAIL,
                    Subject: `Lembrete de Pagamento de Cota`,
                    HtmlBody: `<h1>Pagamento de Cota do Mês de ${(0, main_1.getMes)(mesActual)}</h1>
          <p style="font-size: 12pt">
              Saudações ${dadosPessoais.genero === 'F' ? 'Dra.' : 'Dr.'} 
              ${membro.dadosPessoais.nome}, enviamos este email 
              para lhe lembrar de pagar a sua cota de ${(0, main_1.getMes)(mesActual)}.
              Pague as suas cotas e mantem-se como um membro activo.
            </p>
  
            <p style="font-size: 12pt">Use o aplicativo CNP para fazer o pagamento</p>

            <p><i>Este e-mail foi gerado pela plataforma CNP</i></p>
          `,
                    MessageStream: 'outbound',
                });
                return;
            }
        }
    }
};
exports.sendPaymentNotification = sendPaymentNotification;
// Funcao para criar situacao financeira
const createSF = async (ano, mes, membroId) => {
    const membro = await membro_1.default.findById(membroId);
    if (!membro) {
        const error = new Error('Membro não encontrado!');
        error.statusCode = 422;
        throw error;
    }
    const sf = await situacao_financeira_1.default.findOne({
        ano: ano,
        mes: mes,
        membro: membroId,
    });
    if (sf) {
        const error = new Error(`Sistuação financeira já existe:  ${(0, main_1.getMes)(mes)} de ${ano}`);
        error.statusCode = 422;
        throw error;
    }
    const situacaoFinanceira = new situacao_financeira_1.default({
        ano: ano,
        mes: mes,
        membro: membroId,
    });
    return situacaoFinanceira.save();
};
exports.createSF = createSF;
// Funcao para verificar sutuacao financeira do mes em curso
const checkMonthlySF = async () => {
    const data = new Date((0, main_1.getAngolaTime)());
    const dia = data.getDate();
    const mes = data.getMonth() + 1;
    const ano = data.getFullYear();
    const hora = data.getHours();
    const minutos = data.getMinutes();
    const segundos = data.getSeconds();
    if (dia === 1 && hora === 0 && minutos === 0 && segundos === 0) {
        await (0, exports.checkAllMembrosSF)(mes, ano);
    }
};
exports.checkMonthlySF = checkMonthlySF;
// Funcao para criar sutuacao financeiras de todos os membros no anualmente
const createAnnuallySF = async () => {
    const data = new Date((0, main_1.getAngolaTime)());
    const dia = data.getDate();
    const mes = data.getMonth() + 1;
    const ano = data.getFullYear();
    const hora = data.getHours();
    const minutos = data.getMinutes();
    const segundos = data.getSeconds();
    let counter = 0;
    if (mes === 1 && dia === 1 && hora === 0 && minutos === 0 && segundos === 0) {
        const membros = await membro_1.default.find({
            $and: [{ status: { $ne: 'Suspenso' } }, { status: { $ne: 'Falecido' } }],
        });
        for await (let membro of membros) {
            for (let m = 1; m <= 12; m++) {
                const situacaoFinanceira = await situacao_financeira_1.default.findOne({
                    ano: ano,
                    mes: m,
                    membro: membro.id,
                });
                if (!situacaoFinanceira) {
                    const s = await situacao_financeira_1.default.create({
                        ano: ano,
                        mes: m,
                        membro: membro.id,
                    });
                    console.log(s);
                }
            }
            console.log('Counter: ', counter++);
        }
    }
};
exports.createAnnuallySF = createAnnuallySF;
const checkReferences = async () => {
    const data = new Date((0, main_1.getAngolaTime)());
    if (!runningCheckReference) {
        runningCheckReference = true;
        const referencias = await referencia_1.default.find({
            status: 'Pendente',
            dataExpiracao: { $lt: data },
        });
        for (let referencia of referencias) {
            const pagamento = await pagamento_1.default.findOne({
                status: 'Pendente',
                referencia: referencia.id,
            });
            if (pagamento) {
                const membro = await membro_1.default.findOne({
                    dadosPessoais: pagamento.dadosPessoais,
                });
                const situacaoFinanceira = await situacao_financeira_1.default.findOne({
                    mes: pagamento.mes,
                    ano: pagamento.ano,
                    membro: membro === null || membro === void 0 ? void 0 : membro.id,
                });
                if (situacaoFinanceira) {
                    situacaoFinanceira.status = 'Devendo';
                    await situacaoFinanceira.save();
                }
                await pagamento.delete();
            }
            referencia.status = 'Expirada';
            await referencia.save();
        }
        runningCheckReference = false;
    }
};
exports.checkReferences = checkReferences;
const addFecho = async (pagamento) => {
    const mes = pagamento.updatedAt.getMonth() + 1;
    const ano = pagamento.updatedAt.getFullYear();
    const fecho = await fecho_1.default.findOne({ mes: mes, ano: ano });
    if (!fecho) {
        await fecho_1.default.create({
            mes,
            ano,
            totalEntrada: pagamento.valorPago,
            saldo: pagamento.valorPago,
            emolumentos: {
                [pagamento.emolumento.designacao]: +pagamento.valorPago,
            },
        });
    }
    else {
        fecho.totalEntrada += +pagamento.valorPago;
        fecho.saldo = fecho.totalEntrada - fecho.totalSaida;
        await fecho.save();
        const emolumento = { ...fecho.emolumentos };
        // Checking emolumento
        if (pagamento.emolumento.designacao in emolumento) {
            emolumento[pagamento.emolumento.designacao] += +pagamento.valorPago;
            await fecho.updateOne({ $set: { emolumentos: emolumento } });
        }
        else {
            emolumento[pagamento.emolumento.designacao] = +pagamento.valorPago;
            await fecho.updateOne({ $set: { emolumentos: emolumento } });
        }
    }
};
exports.addFecho = addFecho;
